import { Injectable } from '@angular/core';
import { HttpClient } from '@angular/common/http';
import { Observable } from 'rxjs';

// Interfaces for our data
export interface Match {
  matchId: string;
  teams: string[];
  date: string;
  venue: string;
  winner: string;
  createdAt: string;
}

export interface MatchDetails {
  matchInfo: {
    teams: string[];
    date: string;
    venue: string;
    city: string;
    matchType: string;
    overs: number;
    winner: string;
    result: any;
    playerOfMatch: string[];
    toss: any;
    officials: any;
    event: any;
    season: string;
  };
  innings: Inning[];
}

export interface FallOfWicket {
  player: string;
  score: number;
  over: string;
}

export interface Inning {
  team: string;
  totalRuns: number;
  totalWickets: number;
  totalOvers: string;
  target?: any;
  extras: {
    total: number;
    wides: number;
    noBalls: number;
    byes: number;
    legByes: number;
  };
  battingStats: PlayerStats[];
  bowlingStats: BowlingStats[];
  powerplays: any[];
  fallOfWickets?: FallOfWicket[];
}

export interface PlayerStats {
  player: string;
  runs: number;
  balls: number;
  fours: number;
  sixes: number;
  strikeRate: string;
  out: boolean;
  dismissal?: any;
}

export interface BowlingStats {
  bowler: string;
  overs: string;
  runs: number;
  wickets: number;
  economy: string;
}

@Injectable({
  providedIn: 'root',
})
export class CricketService {
  private apiUrl = 'https://cricket-analytics-backendd.onrender.com/api';

  constructor(private http: HttpClient) {}

  // Get all matches
  getAllMatches(): Observable<{ success: boolean; data: Match[] }> {
    return this.http.get<{ success: boolean; data: Match[] }>(`${this.apiUrl}/matches`);
  }

  // Get match details by ID
  getMatchDetails(matchId: string): Observable<{ success: boolean; data: MatchDetails }> {
    return this.http.get<{ success: boolean; data: MatchDetails }>(
      `${this.apiUrl}/getMatchInfo/${matchId}`
    );
  }

  // Add new match
  addMatch(matchData: any): Observable<{ success: boolean; message: string; matchId: string }> {
    return this.http.post<{ success: boolean; message: string; matchId: string }>(
      `${this.apiUrl}/addMatchInfo`,
      matchData
    );
  }

  // Get raw match data
  getRawMatchData(matchId: string): Observable<{ success: boolean; data: any }> {
    return this.http.get<{ success: boolean; data: any }>(`${this.apiUrl}/rawMatch/${matchId}`);
  }
}
