import { Component, Input, OnInit, AfterViewInit, OnChanges } from '@angular/core';
import { initFlowbite } from 'flowbite';
import { MatchDetails as MatchDetailsInterface, FallOfWicket } from '../../services/cricket';

@Component({
  selector: 'app-match-visualization',
  standalone: false,
  templateUrl: './match-visualization.html',
  styleUrl: './match-visualization.css',
})
export class MatchVisualization implements OnInit, AfterViewInit, OnChanges {
  @Input() matchDetails: MatchDetailsInterface | null = null;

  // Get top 5 batsmen across all innings
  getTopBatsmen() {
    if (!this.matchDetails) return [];

    const allBatsmen: any[] = [];
    this.matchDetails.innings.forEach((inning) => {
      if (inning.battingStats) {
        allBatsmen.push(...inning.battingStats);
      }
    });

    return allBatsmen.sort((a, b) => b.runs - a.runs).slice(0, 5);
  }

  // Get top 5 bowlers across all innings
  getTopBowlers() {
    if (!this.matchDetails) return [];

    const allBowlers: any[] = [];
    this.matchDetails.innings.forEach((inning) => {
      if (inning.bowlingStats) {
        allBowlers.push(...inning.bowlingStats);
      }
    });

    return allBowlers.sort((a, b) => b.wickets - a.wickets).slice(0, 5);
  }

  // Helper method to calculate run rate
  calculateRunRate(runs: number, overs: string): number {
    const oversFloat = parseFloat(overs);
    return oversFloat > 0 ? runs / oversFloat : 0;
  }

  // Dashboard Card Methods
  getTotalRuns(): number {
    if (!this.matchDetails) return 0;
    return this.matchDetails.innings.reduce((total, inning) => total + inning.totalRuns, 0);
  }

  getTotalWickets(): number {
    if (!this.matchDetails) return 0;
    return this.matchDetails.innings.reduce((total, inning) => total + inning.totalWickets, 0);
  }

  getAverageRunRate(): number {
    if (!this.matchDetails) return 0;
    const totalRuns = this.getTotalRuns();
    const totalOvers = this.matchDetails.innings.reduce((total, inning) => {
      return total + parseFloat(inning.totalOvers);
    }, 0);
    return totalOvers > 0 ? totalRuns / totalOvers : 0;
  }

  getTotalExtras(): number {
    if (!this.matchDetails) return 0;
    return this.matchDetails.innings.reduce((total, inning) => total + inning.extras.total, 0);
  }

  // Chart Data Methods
  getTeamRunsChartData(): any {
    if (!this.matchDetails) return {};
    return {
      labels: this.matchDetails.innings.map((inning) => inning.team),
      datasets: [
        {
          label: 'Runs',
          data: this.matchDetails.innings.map((inning) => inning.totalRuns),
          backgroundColor: ['#667eea', '#f093fb', '#4facfe', '#43e97b', '#fa709a', '#ffecd2'],
          borderColor: [this.getThemeColor('primary'), this.getThemeColor('secondary')],
          borderWidth: 2,
        },
      ],
    };
  }

  getExtrasChartData(): any {
    if (!this.matchDetails) return {};
    const totalExtras = {
      wides: this.matchDetails.innings.reduce((sum, inning) => sum + inning.extras.wides, 0),
      noBalls: this.matchDetails.innings.reduce((sum, inning) => sum + inning.extras.noBalls, 0),
      byes: this.matchDetails.innings.reduce((sum, inning) => sum + inning.extras.byes, 0),
      legByes: this.matchDetails.innings.reduce((sum, inning) => sum + inning.extras.legByes, 0),
    };
    return {
      labels: ['Wides', 'No Balls', 'Byes', 'Leg Byes'],
      datasets: [
        {
          data: [totalExtras.wides, totalExtras.noBalls, totalExtras.byes, totalExtras.legByes],
          backgroundColor: ['#ff6b6b', '#4ecdc4', '#45b7d1', '#96ceb4', '#feca57', '#ff9ff3'],
          borderWidth: 2,
          borderColor: this.getThemeColor('base-100'),
        },
      ],
    };
  }

  getTopBatsmenChartData(): any {
    if (!this.matchDetails) return {};
    const topBatsmen = this.getTopBatsmen();
    return {
      labels: topBatsmen.map((player) => player.player),
      datasets: [
        {
          data: topBatsmen.map((player) => player.runs),
          backgroundColor: ['#667eea', '#f093fb', '#4facfe', '#43e97b', '#fa709a', '#ffecd2'],
          borderWidth: 2,
          borderColor: this.getThemeColor('base-100'),
        },
      ],
    };
  }

  getRunRateChartData(): any {
    if (!this.matchDetails) return {};

    const labels = ['5 overs', '10 overs', '15 overs', '20 overs'];
    const datasets = this.matchDetails.innings.map((inning, index) => {
      const runRate = this.calculateRunRate(inning.totalRuns, inning.totalOvers);
      return {
        label: inning.team,
        data: [runRate * 0.5, runRate * 0.7, runRate * 0.9, runRate], // Simulated progression
        borderColor: index === 0 ? '#667eea' : '#f093fb',
        backgroundColor: index === 0 ? 'rgba(102, 126, 234, 0.1)' : 'rgba(240, 147, 251, 0.1)',
        fill: false,
        tension: 0.4,
      };
    });

    return {
      labels,
      datasets,
    };
  }

  private getWicketsTimeline(): { over: string; wicketNumber: number }[] {
    if (!this.matchDetails) return [];
    const timeline: { over: string; wicketNumber: number }[] = [];
    let wicketCount = 0;

    this.matchDetails.innings.forEach((inning) => {
      if (inning.fallOfWickets) {
        inning.fallOfWickets.forEach((wicket: FallOfWicket) => {
          wicketCount++;
          timeline.push({
            over: wicket.over,
            wicketNumber: wicketCount,
          });
        });
      }
    });

    return timeline;
  }

  ngOnInit() {
    // Component initialization
  }

  private getThemeColor(colorName: string, opacity: number = 1): string {
    // Get computed style from document root to access CSS custom properties
    const root = document.documentElement;
    const computedStyle = getComputedStyle(root);

    // Map DaisyUI color names to CSS custom properties
    const colorMap: { [key: string]: string } = {
      'base-content': '--bc',
      'base-100': '--b1',
      'base-200': '--b2',
      'base-300': '--b3',
      primary: '--p',
      secondary: '--s',
      accent: '--a',
      neutral: '--n',
    };

    const cssVar = colorMap[colorName] || '--bc';
    const colorValue = computedStyle.getPropertyValue(cssVar).trim();

    if (colorValue) {
      // Convert hex to rgba if opacity is specified
      if (opacity < 1 && colorValue.startsWith('#')) {
        const hex = colorValue.replace('#', '');
        const r = parseInt(hex.substr(0, 2), 16);
        const g = parseInt(hex.substr(2, 2), 16);
        const b = parseInt(hex.substr(4, 2), 16);
        return `rgba(${r}, ${g}, ${b}, ${opacity})`;
      }
      return colorValue;
    }

    // Fallback colors
    const fallbackColors: { [key: string]: string } = {
      'base-content': '#1F2937',
      'base-100': '#ffffff',
      'base-200': '#f3f4f6',
      'base-300': '#d1d5db',
      primary: '#3B82F6',
      secondary: '#EF4444',
      accent: '#10B981',
      neutral: '#6B7280',
    };

    return fallbackColors[colorName] || '#1F2937';
  }

  ngAfterViewInit() {
    // Initialize Flowbite charts after view is ready
    setTimeout(() => {
      this.initializeCharts();
    }, 100);
  }

  ngOnChanges() {
    // Re-initialize charts when data changes
    if (this.matchDetails) {
      setTimeout(() => {
        this.initializeCharts();
      }, 100);
    }
  }

  private initializeCharts() {
    if (!this.matchDetails) return;

    // Initialize Flowbite charts
    initFlowbite();

    // Initialize individual charts
    this.initializeTeamRunsChart();
    this.initializeExtrasChart();
    this.initializeBatsmenChart();
    this.initializeRunRateChart();
  }

  private initializeTeamRunsChart() {
    const canvas = document.getElementById('team-runs-chart') as HTMLCanvasElement;
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    if (!ctx) return;

    // Clear canvas
    ctx.clearRect(0, 0, canvas.width, canvas.height);

    const data = this.getTeamRunsChartData();

    // Simple bar chart implementation
    const maxValue = Math.max(...data.datasets[0].data);
    const padding = 60;
    const chartWidth = canvas.width - padding * 2;
    const chartHeight = canvas.height - padding * 2;
    const barWidth = chartWidth / data.labels.length;

    data.labels.forEach((label: string, index: number) => {
      const value = data.datasets[0].data[index];
      const barHeight = (value / maxValue) * chartHeight;
      const x = padding + index * barWidth + 10;
      const y = canvas.height - padding - barHeight;

      // Draw bar
      ctx.fillStyle = data.datasets[0].backgroundColor[index];
      ctx.fillRect(x, y, barWidth - 20, barHeight);

      // Draw bar border
      ctx.strokeStyle = this.getThemeColor('base-content');
      ctx.lineWidth = 1;
      ctx.strokeRect(x, y, barWidth - 20, barHeight);

      // Draw label
      ctx.fillStyle = this.getThemeColor('base-content', 0.7);
      ctx.font = '14px Arial';
      ctx.textAlign = 'center';
      ctx.fillText(label, x + (barWidth - 20) / 2, canvas.height - 20);

      // Draw value
      ctx.fillStyle = this.getThemeColor('base-content');
      ctx.font = 'bold 16px Arial';
      ctx.textAlign = 'center';
      ctx.fillText(value.toString(), x + (barWidth - 20) / 2, y - 10);
    });
  }

  private initializeExtrasChart() {
    const canvas = document.getElementById('extras-chart') as HTMLCanvasElement;
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    if (!ctx) return;

    // Clear canvas
    ctx.clearRect(0, 0, canvas.width, canvas.height);

    const data = this.getExtrasChartData();
    const total = data.datasets[0].data.reduce((sum: number, val: number) => sum + val, 0);

    if (total === 0) return;

    let currentAngle = 0;
    const centerX = canvas.width / 2;
    const centerY = canvas.height / 2 - 40; // Move up more for better alignment
    const radius = Math.min(centerX, centerY) - 80; // Smaller radius for more space

    data.labels.forEach((label: string, index: number) => {
      const value = data.datasets[0].data[index];
      const sliceAngle = (value / total) * 2 * Math.PI;

      // Draw slice
      ctx.beginPath();
      ctx.moveTo(centerX, centerY);
      ctx.arc(centerX, centerY, radius, currentAngle, currentAngle + sliceAngle);
      ctx.closePath();
      ctx.fillStyle = data.datasets[0].backgroundColor[index];
      ctx.fill();

      // Draw slice border
      ctx.strokeStyle = '#ffffff';
      ctx.lineWidth = 2;
      ctx.stroke();

      currentAngle += sliceAngle;
    });

    // Draw legend below the chart with better spacing
    const legendY = centerY + radius + 60;
    const legendX = centerX - (data.labels.length * 100) / 2; // More spacing between items

    data.labels.forEach((label: string, index: number) => {
      const value = data.datasets[0].data[index];
      const x = legendX + index * 100; // Increased spacing

      // Draw legend color box
      ctx.fillStyle = data.datasets[0].backgroundColor[index];
      ctx.fillRect(x - 15, legendY - 8, 12, 12);

      // Draw legend text
      ctx.fillStyle = '#374151';
      ctx.font = 'bold 12px Arial';
      ctx.textAlign = 'left';
      ctx.fillText(`${label}: ${value}`, x + 5, legendY + 4);
    });
  }

  private initializeBatsmenChart() {
    const canvas = document.getElementById('batsmen-chart') as HTMLCanvasElement;
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    if (!ctx) return;

    // Clear canvas
    ctx.clearRect(0, 0, canvas.width, canvas.height);

    const data = this.getTopBatsmenChartData();
    const total = data.datasets[0].data.reduce((sum: number, val: number) => sum + val, 0);

    if (total === 0) return;

    let currentAngle = 0;
    const centerX = canvas.width / 2;
    const centerY = canvas.height / 2 - 20; // Move up slightly for better alignment
    const radius = Math.min(centerX, centerY) - 80;

    data.labels.forEach((label: string, index: number) => {
      const value = data.datasets[0].data[index];
      const sliceAngle = (value / total) * 2 * Math.PI;

      // Draw slice
      ctx.beginPath();
      ctx.moveTo(centerX, centerY);
      ctx.arc(centerX, centerY, radius, currentAngle, currentAngle + sliceAngle);
      ctx.closePath();
      ctx.fillStyle = data.datasets[0].backgroundColor[index];
      ctx.fill();

      // Draw slice border
      ctx.strokeStyle = '#ffffff';
      ctx.lineWidth = 2;
      ctx.stroke();

      // Draw label
      const labelAngle = currentAngle + sliceAngle / 2;
      const labelX = centerX + Math.cos(labelAngle) * (radius + 50);
      const labelY = centerY + Math.sin(labelAngle) * (radius + 50);
      ctx.fillStyle = '#374151';
      ctx.font = 'bold 12px Arial';
      ctx.textAlign = 'center';
      ctx.fillText(`${label}: ${value}`, labelX, labelY);

      currentAngle += sliceAngle;
    });
  }

  private initializeRunRateChart() {
    const canvas = document.getElementById('run-rate-chart') as HTMLCanvasElement;
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    if (!ctx) return;

    // Clear canvas
    ctx.clearRect(0, 0, canvas.width, canvas.height);

    const data = this.getRunRateChartData();

    if (data.labels.length === 0) return;

    const maxValue = Math.max(...data.datasets[0].data, ...data.datasets[1].data);
    const minValue = Math.min(...data.datasets[0].data, ...data.datasets[1].data);
    const padding = 60;
    const chartWidth = canvas.width - padding * 2;
    const chartHeight = canvas.height - padding * 2;
    const stepX = chartWidth / (data.labels.length - 1);
    const stepY = chartHeight / (maxValue - minValue);

    // Draw grid lines
    ctx.strokeStyle = '#E5E7EB';
    ctx.lineWidth = 1;
    for (let i = 0; i <= 5; i++) {
      const value = minValue + (maxValue - minValue) * (i / 5);
      const y = canvas.height - padding - (value - minValue) * stepY;
      ctx.beginPath();
      ctx.moveTo(padding, y);
      ctx.lineTo(canvas.width - padding, y);
      ctx.stroke();

      // Draw grid labels
      ctx.fillStyle = '#6B7280';
      ctx.font = '10px Arial';
      ctx.textAlign = 'right';
      ctx.fillText(value.toFixed(1), padding - 10, y + 3);
    }

    // Draw team lines
    const colors = ['#3B82F6', '#EF4444']; // Blue and Red
    const teamNames = this.matchDetails?.innings.map((inning) => inning.team) || [];

    data.datasets.forEach((dataset: any, datasetIndex: number) => {
      ctx.strokeStyle = colors[datasetIndex];
      ctx.lineWidth = 3;
      ctx.beginPath();

      data.labels.forEach((label: string, index: number) => {
        const value = dataset.data[index];
        const x = padding + index * stepX;
        const y = canvas.height - padding - (value - minValue) * stepY;

        if (index === 0) {
          ctx.moveTo(x, y);
        } else {
          ctx.lineTo(x, y);
        }

        // Draw label
        ctx.fillStyle = '#374151';
        ctx.font = 'bold 12px Arial';
        ctx.textAlign = 'center';
        ctx.fillText(label, x, canvas.height - 20);
      });

      ctx.stroke();
    });

    // Draw legend
    const legendY = 30;
    teamNames.forEach((team: string, index: number) => {
      const x = padding + index * 150;

      // Draw legend color line
      ctx.strokeStyle = colors[index];
      ctx.lineWidth = 3;
      ctx.beginPath();
      ctx.moveTo(x, legendY);
      ctx.lineTo(x + 20, legendY);
      ctx.stroke();

      // Draw legend text
      ctx.fillStyle = '#374151';
      ctx.font = 'bold 14px Arial';
      ctx.textAlign = 'left';
      ctx.fillText(team, x + 25, legendY + 4);
    });
  }
}
