import { Component } from '@angular/core';
import { Router } from '@angular/router';
import { CricketService } from '../../services/cricket';

@Component({
  selector: 'app-match-upload',
  standalone: false,
  templateUrl: './match-upload.html',
  styleUrl: './match-upload.css',
})
export class MatchUpload {
  isDragOver = false;
  isUploading = false;
  uploadStatus = '';
  uploadError = '';
  uploadedMatchId = '';

  constructor(private cricketService: CricketService, private router: Router) {}

  onDragOver(event: DragEvent) {
    event.preventDefault();
    this.isDragOver = true;
  }

  onDragLeave(event: DragEvent) {
    event.preventDefault();
    this.isDragOver = false;
  }

  onDrop(event: DragEvent) {
    event.preventDefault();
    this.isDragOver = false;

    const files = event.dataTransfer?.files;
    if (files && files.length > 0) {
      this.handleFile(files[0]);
    }
  }

  onFileSelected(event: any) {
    const file = event.target.files[0];
    if (file) {
      this.handleFile(file);
    }
  }

  private handleFile(file: File) {
    // Validate file type
    if (file.type !== 'application/json' && !file.name.endsWith('.json')) {
      this.uploadError = 'Please select a JSON file';
      return;
    }

    // Validate file size (max 10MB)
    if (file.size > 10 * 1024 * 1024) {
      this.uploadError = 'File size must be less than 10MB';
      return;
    }

    this.uploadFile(file);
  }

  private uploadFile(file: File) {
    this.isUploading = true;
    this.uploadError = '';
    this.uploadStatus = 'Reading file...';

    const reader = new FileReader();
    reader.onload = (e) => {
      try {
        const jsonData = JSON.parse(e.target?.result as string);
        this.uploadStatus = 'Uploading match data...';

        this.cricketService.addMatch(jsonData).subscribe({
          next: (response) => {
            this.isUploading = false;
            this.uploadStatus = 'Upload successful!';
            this.uploadedMatchId = response.matchId;

            // Clear form after 3 seconds
            setTimeout(() => {
              this.resetForm();
            }, 3000);
          },
          error: (error) => {
            this.isUploading = false;
            this.uploadError =
              error.error?.message || 'Upload failed. Please check your JSON format.';
            this.uploadStatus = '';
          },
        });
      } catch (error) {
        this.isUploading = false;
        this.uploadError = 'Invalid JSON format. Please check your file.';
        this.uploadStatus = '';
      }
    };

    reader.onerror = () => {
      this.isUploading = false;
      this.uploadError = 'Error reading file';
      this.uploadStatus = '';
    };

    reader.readAsText(file);
  }

  private resetForm() {
    this.uploadStatus = '';
    this.uploadError = '';
    this.uploadedMatchId = '';
    this.isUploading = false;
  }

  viewMatch() {
    if (this.uploadedMatchId) {
      this.router.navigate(['/match', this.uploadedMatchId]);
    }
  }
}
