import { Component, OnInit } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { CricketService, MatchDetails as MatchDetailsInterface } from '../../services/cricket';

@Component({
  selector: 'app-match-details',
  templateUrl: './match-details.html',
  styleUrls: ['./match-details.css'],
  standalone: false,
})
export class MatchDetails implements OnInit {
  matchId: string = '';
  matchDetails: MatchDetailsInterface | null = null;
  loading = true;
  error = '';
  selectedInningIndex = 0; // Track which inning tab is selected

  constructor(
    private cricketService: CricketService,
    private route: ActivatedRoute,
    private router: Router
  ) {}

  // Generate avatar URL for player names
  getPlayerAvatar(playerName: string): string {
    // Split name and take first two words (first name + last name)
    const nameParts = playerName.trim().split(' ');
    let avatarName = '';

    if (nameParts.length >= 2) {
      // Use first and last name
      avatarName = `${nameParts[0]}+${nameParts[nameParts.length - 1]}`;
    } else if (nameParts.length === 1) {
      // Use single name
      avatarName = nameParts[0];
    } else {
      // Fallback
      avatarName = 'Player';
    }

    return `https://ui-avatars.com/api/?name=${avatarName}&size=40&background=ededed&color=374151&rounded=true`;
  }

  // Get team flag URL based on team name
  getTeamFlag(teamName: string): string {
    // Map team names to country codes for flag API
    const teamFlagMap: { [key: string]: string } = {
      Bangladesh: 'bd',
      Pakistan: 'pk',
      India: 'in',
      Australia: 'au',
      England: 'gb',
      'New Zealand': 'nz',
      'South Africa': 'za',
      'Sri Lanka': 'lk',
      'West Indies': 'ag', // Using Antigua as representative
      Afghanistan: 'af',
      Ireland: 'ie',
      Scotland: 'gb-sct',
      Netherlands: 'nl',
      Zimbabwe: 'zw',
      Kenya: 'ke',
      Canada: 'ca',
      USA: 'us',
    };

    // Get country code from team name (case insensitive)
    const countryCode = teamFlagMap[teamName] || 'xx'; // 'xx' for unknown teams

    return `https://flagcdn.com/${countryCode}.svg`;
  }

  // Get player's team flag by finding which team the player belongs to
  getPlayerTeamFlag(playerName: string): string {
    if (!this.matchDetails || !this.matchDetails.innings) {
      return this.getTeamFlag('Unknown');
    }

    // Search through all innings to find which team the player belongs to
    for (const inning of this.matchDetails.innings) {
      // Check batting stats
      if (inning.battingStats) {
        for (const player of inning.battingStats) {
          if (player.player === playerName) {
            return this.getTeamFlag(inning.team);
          }
        }
      }

      // Check bowling stats
      if (inning.bowlingStats) {
        for (const bowler of inning.bowlingStats) {
          if (bowler.bowler === playerName) {
            return this.getTeamFlag(inning.team);
          }
        }
      }
    }

    // If player not found, return unknown flag
    return this.getTeamFlag('Unknown');
  }

  // Get fall of wickets information
  getFallOfWickets(inning: any): any[] {
    if (!inning || !inning.fallOfWickets) {
      return [];
    }

    return inning.fallOfWickets;
  }

  // Get players who did not bat
  getDidNotBat(inning: any): string[] {
    if (!inning || !inning.battingStats) {
      return [];
    }

    // Get all players who batted (have batting stats)
    const battedPlayers = inning.battingStats.map((player: any) => player.player);

    // Get all players in the team (from bowling stats and other sources)
    const allTeamPlayers: string[] = [];

    // Add players from batting stats
    inning.battingStats.forEach((player: any) => {
      if (!allTeamPlayers.includes(player.player)) {
        allTeamPlayers.push(player.player);
      }
    });

    // Add players from bowling stats (opposing team)
    if (inning.bowlingStats) {
      inning.bowlingStats.forEach((bowler: any) => {
        if (!allTeamPlayers.includes(bowler.bowler)) {
          allTeamPlayers.push(bowler.bowler);
        }
      });
    }

    // Find players who didn't bat
    const didNotBat = allTeamPlayers.filter((player) => !battedPlayers.includes(player));

    return didNotBat;
  }

  loadMatchDetails() {
    if (!this.matchId) {
      this.error = 'No match ID provided';
      this.loading = false;
      return;
    }

    this.loading = true;
    this.error = '';

    this.cricketService.getMatchDetails(this.matchId).subscribe({
      next: (response) => {
        if (response.success) {
          this.matchDetails = response.data;
        } else {
          this.error = 'Failed to load match details';
        }
        this.loading = false;
      },
      error: (err) => {
        this.error = 'Error loading match details: ' + err.message;
        this.loading = false;
      },
    });
  }

  ngOnInit() {
    // Get match ID from route parameters
    this.route.params.subscribe((params) => {
      this.matchId = params['id'];
      this.loadMatchDetails();
    });
  }

  goBack() {
    this.router.navigate(['/matches']);
  }

  viewRawJSON() {
    if (this.matchId) {
      this.cricketService.getRawMatchData(this.matchId).subscribe({
        next: (response) => {
          if (response.success) {
            // Open raw JSON in a new window/tab
            const jsonWindow = window.open('', '_blank');
            if (jsonWindow) {
              jsonWindow.document.write(`
                <html>
                  <head>
                    <title>Raw Match JSON - ${this.matchId}</title>
                    <style>
                      body { font-family: monospace; margin: 20px; background: #f5f5f5; }
                      pre { background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
                      .header { background: #4f46e5; color: white; padding: 15px; border-radius: 8px; margin-bottom: 20px; }
                    </style>
                  </head>
                  <body>
                    <div class="header">
                      <h2>Raw Match JSON Data</h2>
                      <p>Match ID: ${this.matchId}</p>
                    </div>
                    <pre>${JSON.stringify(response.data, null, 2)}</pre>
                  </body>
                </html>
              `);
              jsonWindow.document.close();
            }
          }
        },
        error: (err) => {
          alert('Error loading raw JSON data: ' + err.message);
        },
      });
    }
  }
}
