const express = require('express');
const morgan = require('morgan');
const cors = require('cors');
require('dotenv').config();

// Import routers
const matchRouter = require('./routers/matchRouter');

const app = express();
const PORT = process.env.PORT || 3000;

// Middleware
app.use(cors());
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Morgan logging
app.use(morgan('combined'));

// Routes
app.use('/api', matchRouter);

// Basic route
app.get('/', (req, res) => {
    res.json({
        message: 'Cricket Match Data API',
        endpoints: {
            'POST /api/addMatchInfo': 'Add new match data',
            'GET /api/getMatchInfo/:matchId': 'Get match scorecard',
            'GET /api/matches': 'Get all matches'
        }
    });
});

// Error handling middleware
app.use((err, req, res, next) => {
    console.error(err.stack);
    res.status(500).json({
        message: 'Something went wrong!',
        error: process.env.NODE_ENV === 'development' ? err.message : 'Internal Server Error'
    });
});

// 404 handler
app.use('*', (req, res) => {
    res.status(404).json({
        message: 'Route not found',
        path: req.originalUrl
    });
});

// Start server
app.listen(PORT, () => {
    console.log(`🚀 Cricket API Server running on port ${PORT}`);
    console.log(`🌐 Server URL: źhttp://localhost:${PORT}`);
});

module.exports = app;
