const fs = require('fs').promises;
const path = require('path');
const { v4: uuidv4 } = require('uuid');

// File path for storing match data
const MATCHES_FILE = path.join(__dirname, '../data/matches.json');

// Ensure data directory exists
const ensureDataDirectory = async () => {
    const dataDir = path.dirname(MATCHES_FILE);
    try {
        await fs.access(dataDir);
    } catch {
        await fs.mkdir(dataDir, { recursive: true });
    }
};

// Initialize matches file if it doesn't exist
const initializeMatchesFile = async () => {
    try {
        await fs.access(MATCHES_FILE);
    } catch {
        await fs.writeFile(MATCHES_FILE, JSON.stringify([]));
    }
};

// Save match data
const saveMatch = async (matchData) => {
    try {
        await ensureDataDirectory();
        await initializeMatchesFile();
        
        // Generate unique match ID
        const matchId = uuidv4();
        const matchWithId = {
            matchId,
            ...matchData,
            createdAt: new Date().toISOString()
        };
        
        // Read existing matches
        const existingMatches = await getAllMatches();
        
        // Add new match
        existingMatches.push(matchWithId);
        
        // Save back to file
        await fs.writeFile(MATCHES_FILE, JSON.stringify(existingMatches, null, 2));
        
        return matchId;
    } catch (error) {
        throw new Error(`Failed to save match: ${error.message}`);
    }
};

// Get all matches
const getAllMatches = async () => {
    try {
        await ensureDataDirectory();
        await initializeMatchesFile();
        
        const data = await fs.readFile(MATCHES_FILE, 'utf8');
        return JSON.parse(data);
    } catch (error) {
        throw new Error(`Failed to get matches: ${error.message}`);
    }
};

// Get match by ID
const getMatchById = async (matchId) => {
    try {
        const matches = await getAllMatches();
        const match = matches.find(m => m.matchId === matchId);
        
        if (!match) {
            throw new Error(`Match with ID ${matchId} not found`);
        }
        
        return match;
    } catch (error) {
        throw new Error(`Failed to get match: ${error.message}`);
    }
};

// Delete match by ID
const deleteMatch = async (matchId) => {
    try {
        const matches = await getAllMatches();
        const filteredMatches = matches.filter(m => m.matchId !== matchId);
        
        if (filteredMatches.length === matches.length) {
            throw new Error(`Match with ID ${matchId} not found`);
        }
        
        await fs.writeFile(MATCHES_FILE, JSON.stringify(filteredMatches, null, 2));
        return true;
    } catch (error) {
        throw new Error(`Failed to delete match: ${error.message}`);
    }
};

module.exports = {
    saveMatch,
    getAllMatches,
    getMatchById,
    deleteMatch
};
