const matchModel = require('../models/matchModel');

// Add match info controller
const addMatchInfo = async (req, res) => {
    try {
        const matchData = req.body;
        
        // Basic validation
        if (!matchData.info || !matchData.innings || !matchData.meta) {
            return res.status(400).json({
                success: false,
                message: 'Invalid match data. Required fields: info, innings, meta'
            });
        }
        
        // Save match data
        const matchId = await matchModel.saveMatch(matchData);
        
        res.status(201).json({
            success: true,
            message: 'Match data saved successfully',
            matchId: matchId
        });
        
    } catch (error) {
        console.error('Error adding match info:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to save match data',
            error: error.message
        });
    }
};

// Get match info controller
const getMatchInfo = async (req, res) => {
    try {
        const { matchId } = req.params;
        
        if (!matchId) {
            return res.status(400).json({
                success: false,
                message: 'Match ID is required'
            });
        }
        
        // Get raw match data
        const rawMatchData = await matchModel.getMatchById(matchId);
        
        // Process the data into scorecard format
        const processedScorecard = processMatchData(rawMatchData);
        
        res.status(200).json({
            success: true,
            data: processedScorecard
        });
        
    } catch (error) {
        console.error('Error getting match info:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to get match data',
            error: error.message
        });
    }
};

// Get all matches controller
const getAllMatches = async (req, res) => {
    try {
        const matches = await matchModel.getAllMatches();
        
        // Return only basic info for list view
        const matchList = matches.map(match => ({
            matchId: match.matchId,
            teams: match.info.teams,
            date: match.info.dates[0],
            venue: match.info.venue,
            winner: match.info.outcome.winner,
            createdAt: match.createdAt
        }));
        
        res.status(200).json({
            success: true,
            data: matchList
        });
        
    } catch (error) {
        console.error('Error getting all matches:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to get matches',
            error: error.message
        });
    }
};

// Process raw match data into scorecard format
const processMatchData = (rawData) => {
    try {
        const { info, innings, meta } = rawData;
        
        // Comprehensive match info
        const matchInfo = {
            // Basic Info
            teams: info.teams,
            date: info.dates[0],
            venue: info.venue,
            city: info.city,
            matchType: info.match_type,
            matchTypeNumber: info.match_type_number,
            overs: info.overs,
            ballsPerOver: info.balls_per_over,
            gender: info.gender,
            teamType: info.team_type,
            season: info.season,
            
            // Event Details
            event: info.event,
            
            // Match Result
            winner: info.outcome.winner,
            result: info.outcome.by,
            playerOfMatch: info.player_of_match,
            
            // Toss
            toss: info.toss,
            
            // Officials
            officials: info.officials,
            
            // Players
            players: info.players,
            
            // Meta Data
            meta: meta
        };
        
        // Process innings data
        const processedInnings = innings.map(inning => {
            return processInningData(inning);
        });
        
        return {
            matchInfo,
            innings: processedInnings
        };
        
    } catch (error) {
        throw new Error(`Failed to process match data: ${error.message}`);
    }
};

// Process individual inning data
const processInningData = (inning) => {
    try {
        const { team, overs, target, powerplays } = inning;
        
        // Calculate total runs and wickets
        let totalRuns = 0;
        let totalWickets = 0;
        let totalOvers = 0;
        let totalBalls = 0;
        
        // Extras tracking
        let totalExtras = 0;
        let wides = 0;
        let noBalls = 0;
        let byes = 0;
        let legByes = 0;
        
        // Player statistics
        const battingStats = {};
        const bowlingStats = {};
        
        // Fall of wickets tracking
        const fallOfWickets = [];
        let runningScore = 0;
        
        // Process each over
        overs.forEach((over, overIndex) => {
            totalOvers++;
            
            over.deliveries.forEach(delivery => {
                totalBalls++;
                
                // Add runs
                totalRuns += delivery.runs.total;
                totalExtras += delivery.runs.extras;
                
                // Track extras
                if (delivery.extras) {
                    if (delivery.extras.wides) wides += delivery.extras.wides;
                    if (delivery.extras.noballs) noBalls += delivery.extras.noballs;
                    if (delivery.extras.byes) byes += delivery.extras.byes;
                    if (delivery.extras.legbyes) legByes += delivery.extras.legbyes;
                }
                
                // Add wickets
                if (delivery.wickets && delivery.wickets.length > 0) {
                    totalWickets += delivery.wickets.length;
                    
                    // Track bowling wickets
                    if (!bowlingStats[delivery.bowler]) {
                        bowlingStats[delivery.bowler] = {
                            runs: 0,
                            balls: 0,
                            wickets: 0,
                            maidens: 0
                        };
                    }
                    bowlingStats[delivery.bowler].wickets += delivery.wickets.length;
                }
                
                // Track batting stats
                if (!battingStats[delivery.batter]) {
                    battingStats[delivery.batter] = {
                        runs: 0,
                        balls: 0,
                        fours: 0,
                        sixes: 0,
                        out: false,
                        dismissal: null
                    };
                }
                
                battingStats[delivery.batter].runs += delivery.runs.batter;
                battingStats[delivery.batter].balls++;
                
                // Count boundaries
                if (delivery.runs.batter === 4) {
                    battingStats[delivery.batter].fours++;
                } else if (delivery.runs.batter === 6) {
                    battingStats[delivery.batter].sixes++;
                }
                
                // Track if player got out
                if (delivery.wickets && delivery.wickets.length > 0) {
                    delivery.wickets.forEach(wicket => {
                        if (wicket.player_out === delivery.batter) {
                            battingStats[delivery.batter].out = true;
                            battingStats[delivery.batter].dismissal = {
                                kind: wicket.kind,
                                bowler: delivery.bowler,
                                fielders: wicket.fielders || [],
                                fielder: wicket.fielders && wicket.fielders.length > 0 ? 
                                    (typeof wicket.fielders[0] === 'object' ? wicket.fielders[0].name : wicket.fielders[0]) : null
                            };
                            
                            // Record fall of wicket
                            runningScore += delivery.runs.batter;
                            const overNumber = overIndex + 1;
                            const ballNumber = over.deliveries.indexOf(delivery) + 1;
                            const overFormatted = `${overNumber}.${ballNumber}`;
                            
                            fallOfWickets.push({
                                player: delivery.batter,
                                score: runningScore,
                                over: overFormatted
                            });
                        }
                    });
                } else {
                    // Add runs to running score if no wicket
                    runningScore += delivery.runs.batter;
                }
                
                // Track bowling stats
                if (!bowlingStats[delivery.bowler]) {
                    bowlingStats[delivery.bowler] = {
                        runs: 0,
                        balls: 0,
                        wickets: 0,
                        maidens: 0
                    };
                }
                bowlingStats[delivery.bowler].runs += delivery.runs.total;
                bowlingStats[delivery.bowler].balls++;
            });
        });
        
        // Convert balls to overs format (e.g., 20.3 means 20 overs and 3 balls)
        const oversBowled = Math.floor(totalBalls / 6);
        const remainingBalls = totalBalls % 6;
        const oversFormatted = `${oversBowled}.${remainingBalls}`;
        
        // Calculate bowling economy rates
        const processedBowlingStats = Object.entries(bowlingStats).map(([bowler, stats]) => {
            const oversBowled = Math.floor(stats.balls / 6);
            const economy = oversBowled > 0 ? (stats.runs / oversBowled).toFixed(2) : '0.00';
            return {
                bowler,
                ...stats,
                overs: `${Math.floor(stats.balls / 6)}.${stats.balls % 6}`,
                economy: economy
            };
        });
        
        return {
            team,
            totalRuns,
            totalWickets,
            totalOvers: oversFormatted,
            target: target || null,
            powerplays: powerplays || [],
            extras: {
                total: totalExtras,
                wides,
                noBalls,
                byes,
                legByes
            },
            battingStats: Object.entries(battingStats).map(([player, stats]) => ({
                player,
                ...stats,
                strikeRate: stats.balls > 0 ? ((stats.runs / stats.balls) * 100).toFixed(2) : '0.00'
            })),
            bowlingStats: processedBowlingStats,
            fallOfWickets: fallOfWickets
        };
        
    } catch (error) {
        throw new Error(`Failed to process inning data: ${error.message}`);
    }
};

// Get raw match data controller
const getRawMatchData = async (req, res) => {
    try {
        const { matchId } = req.params;
        
        if (!matchId) {
            return res.status(400).json({
                success: false,
                message: 'Match ID is required'
            });
        }
        
        // Get raw match data
        const rawMatchData = await matchModel.getMatchById(matchId);
        
        res.status(200).json({
            success: true,
            data: rawMatchData
        });
        
    } catch (error) {
        console.error('Error getting raw match data:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to get raw match data',
            error: error.message
        });
    }
};

module.exports = {
    addMatchInfo,
    getMatchInfo,
    getAllMatches,
    getRawMatchData
};
